/**  @file bta_avg_sequences.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_avg_sequences.h"
#include <stdlib.h>
#include <string.h>



// Local Prototypes
static BTA_Status average(BTA_Frame **frames, int framesLen, BTA_Frame **frame, BTA_InfoEventInst *infoEventInst);


BTA_Status BFLTavgSequencesInit(BTA_FltAvgSequencesConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    if (config->averageWindowLength < 2) {
        return BTA_StatusInvalidParameter;
    }
    BTA_FltAvgSequencesInst *inst = (BTA_FltAvgSequencesInst *)calloc(1, sizeof(BTA_FltAvgSequencesInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->averageWindowLength = config->averageWindowLength;
    inst->frames = (BTA_Frame **)malloc(inst->averageWindowLength * sizeof(BTA_Frame *));
    if (!inst->frames) {
        free(inst);
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTavgSequencesClose(BTA_FltHandle *handle) {
    BTA_FltAvgSequencesInst **inst = (BTA_FltAvgSequencesInst **)handle;
    free((*inst)->frames);
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTavgSequencesApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_Status status;
    BTA_FltAvgSequencesInst *inst = (BTA_FltAvgSequencesInst *)handle;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }

    inst->frameCounter = (*frame)->frameCounter;
    inst->sequenceCounter = (*frame)->sequenceCounter;
    inst->frameCounterDiff = inst->frameCounter - inst->frameCounterLast;
    if (inst->frameCounterDiff > 0x80000000) {
        // Either a long time gap or a negative frameCounterDiff
        BTAinfoEventHelper(inst->infoEventInst, 5, BTA_StatusInformation, "Filter avg sequences: suspicious frameCounterDiff", inst->frameCounterDiff);
    }

    if (inst->frameCounterDiff > 1) {
        BTAinfoEventHelper(inst->infoEventInst, 10, BTA_StatusInformation, "Filter avg sequences: frames lost:", inst->frameCounterDiff - 1);
    }

    if (inst->sequenceCounter > inst->sequenceCounterLast) {
        inst->sequenceCounterDiff = inst->sequenceCounter - inst->sequenceCounterLast;
        if (inst->sequenceCounterDiff == inst->frameCounterDiff) {
            // We might have missed a sequence, but we are still in the right frame
            inst->frames[inst->framesPos++] = *frame;
            *frame = 0;
            status = BTA_StatusOk;
            if (inst->framesPos == inst->averageWindowLength) {
                // Full. Average and enqueue
                status = average(inst->frames, inst->framesPos, frame, inst->infoEventInst);
                inst->framesPos = 0;
            }
            inst->frameCounterLast = inst->frameCounter;
            inst->sequenceCounterLast = inst->sequenceCounter;
            return status;
        }
    }
    // Starting at sequence 0 again or something was skipped.
    // We are not within the same frame anymore. Average what ever is present and start from framesPos 0
    BTA_Frame *temp;
    status = average(inst->frames, inst->framesPos, &temp, inst->infoEventInst);
    inst->framesPos = 0;
    inst->frames[inst->framesPos++] = *frame;
    *frame = temp;
    inst->frameCounterLast = inst->frameCounter;
    inst->sequenceCounterLast = inst->sequenceCounter;    
    return status;
}


static BTA_Status average(BTA_Frame **frames, int framesLen, BTA_Frame **frame, BTA_InfoEventInst *infoEventInst) {
    int fI;
    int chI;
    int pxI;
    BTA_Channel *ch1, *ch2;
    uint8_t **counters;
    if (!frames) {
        return BTA_StatusInvalidParameter;
    }
    if (!framesLen || !frames) {
        *frame = 0;
        return BTA_StatusInvalidParameter;
    }
    if (framesLen == 1) {
        *frame = frames[0];
        return BTA_StatusOk;
    }
    *frame = frames[0];
    counters = (uint8_t **)malloc((*frame)->channelsLen * sizeof(uint8_t *));
    if (!counters) {
        for (fI = 0; fI < framesLen; fI++) {
            BTAfreeFrame(&frames[fI]);
        }
        return BTA_StatusOutOfMemory;
    }

    // set the counters to 1 (valid pixel) or 0 (invalid pixel) regarding the first frame in frames
    for (chI = 0; chI < (*frame)->channelsLen; chI++) {
        ch1 = (*frame)->channels[chI];
        counters[chI] = 0;
        if (ch1->id == BTA_ChannelIdDistance || ch1->id == BTA_ChannelIdX || ch1->id == BTA_ChannelIdY || ch1->id == BTA_ChannelIdZ) {
            counters[chI] = (uint8_t *)malloc(ch1->xRes * ch1->yRes * sizeof(uint8_t));
            if (!counters[chI]) {
                for (fI = 0; fI < framesLen; fI++) {
                    BTAfreeFrame(&frames[fI]);
                }
                return BTA_StatusOutOfMemory;
            }
            for (pxI = 0; pxI < ch1->xRes * ch1->yRes; pxI++) {
                counters[chI][pxI] = ((float *)ch1->data)[pxI] != 0;
            }
        }
    }

    // Sum up the frames to average
    for (fI = 1; fI < framesLen; fI++) {
        if ((*frame)->channelsLen != frames[fI]->channelsLen) {
            // Number of channels differs, skip frame
            continue;
        }
        for (chI = 0; chI < (*frame)->channelsLen; chI++) {
            ch1 = (*frame)->channels[chI];
            ch2 = frames[fI]->channels[chI];
            if (ch1->xRes != ch2->xRes || ch1->yRes != ch2->yRes || ch1->dataFormat != ch2->dataFormat || ch1->unit != ch2->unit) {
                // Resolution or data differs, skip this channel
                continue;
            }
            if (ch1->id == BTA_ChannelIdDistance || ch1->id == BTA_ChannelIdX || ch1->id == BTA_ChannelIdY || ch1->id == BTA_ChannelIdZ) {
                switch (ch1->dataFormat) {
                case BTA_DataFormatFloat32:
                    for (pxI = 0; pxI < ch1->xRes * ch1->yRes; pxI++) {
                        if (((float *)ch2->data)[pxI] != 0) {
                            ((float *)ch1->data)[pxI] = ((float *)ch1->data)[pxI] + ((float *)ch2->data)[pxI];
                            counters[chI][pxI]++;
                        }
                    }
                    break;
                default:
                    BTAinfoEventHelper(infoEventInst, 1, BTA_StatusNotSupported, "AvgsequenceFilter: DataFormat not implemented", ch1->dataFormat);
                }
            }
            else if (ch1->id == BTA_ChannelIdFlags) {
                switch (ch1->dataFormat) {
                case BTA_DataFormatUInt32:
                    for (pxI = 0; pxI < ch1->xRes * ch1->yRes; pxI++) {
                        ((uint32_t *)ch1->data)[pxI] = ((uint32_t *)ch1->data)[pxI] | ((uint32_t *)ch2->data)[pxI];
                    }
                    break;
                default:
                    BTAinfoEventHelper(infoEventInst, 1, BTA_StatusNotSupported, "AvgsequenceFilter: DataFormat not implemented", ch1->dataFormat);
                }
                //debug
                //((uint32_t *)ch1->data)[0 * ch1->xRes + 0] = framesLen;
                //((uint32_t *)ch1->data)[0 * ch1->xRes + 1] = framesLen;
                //((uint32_t *)ch1->data)[1 * ch1->xRes + 1] = framesLen;
                //((uint32_t *)ch1->data)[1 * ch1->xRes + 0] = framesLen;
            }
        }
    }

    // Divide through counter the summed frames
    for (chI = 0; chI < (*frame)->channelsLen; chI++) {
        BTA_Channel *ch1 = (*frame)->channels[chI];
        if (ch1->id == BTA_ChannelIdDistance || ch1->id == BTA_ChannelIdX || ch1->id == BTA_ChannelIdY || ch1->id == BTA_ChannelIdZ) {
            switch (ch1->dataFormat) {
            case BTA_DataFormatFloat32:
                for (pxI = 0; pxI < ch1->xRes * ch1->yRes; pxI++) {
                    if (counters[chI][pxI] > framesLen >> 1) {
                        ((float *)ch1->data)[pxI] = ((float *)ch1->data)[pxI] / counters[chI][pxI];
                    }
                    else {
                        ((float *)ch1->data)[pxI] = 0;
                    }
                }
                break;
            default:
                BTAinfoEventHelper(infoEventInst, 1, BTA_StatusNotSupported, "AvgsequenceFilter: DataFormat not implemented", ch1->dataFormat);
            }
        }
        free(counters[chI]);
    }
    free(counters);

    // Free all frames except frame 0 (which is returned as result)
    for (fI = 1; fI < framesLen; fI++) {
        BTAfreeFrame(&frames[fI]);
    }
    return BTA_StatusOk;
}